/**
 * \file           CDeviceDriver.h
 * \brief          Header for CDeviceDriver base class
 * \author         RJen
 * \date           03/21/2012
 * \note           All classes found here are intended to be base classes for easier driver implementation
 **/
#ifndef _C_DEVICE_DRIVER_H_
#define _C_DEVICE_DRIVER_H_

// Base class that has a built in locking mechanism
class CDeviceDriver
{
  public:
    CDeviceDriver(void);
    virtual ~CDeviceDriver(void);
    bool BaseInitialize();  // Must be called by derived classes unless overriden  
  protected:
    bool LockDriver(void);    // Locks class     
    bool UnlockDriver(void);  // Unlocks class  
    UINT32 m_hDriverLock;     // Handle to lock
};

// Base class that has a lock and a task that handles interrupt events (for handlers that must block)
class CDeviceDriverIsr : public CDeviceDriver
{
  public:
    CDeviceDriverIsr(void);
    ~CDeviceDriverIsr(void);
    bool BaseInitialize();          // Override the base class method in this instance
    void SignalIsrEvent(void);      // Event to be signaled by ISRs
  protected: 
    // Configure the ISR handler task, must be called only after the derived class is initialized
    bool ConfigureIsrHandlerTask(UINT16 StackSize, UINT32 Priority, const signed char* pTaskName);
    static void IsrHandlerTaskWrapper(UINT32 Param); // C wrapper for IsrHandlerTask()                            
    void IsrHandlerTask(void);      // Task handling the ISR event
    virtual void IsrHandler(void) = 0;  // ISR handler
    UINT32 m_hIsrEvent;             // Handle to the event that should be signaled by ISRs
    UINT32 m_hIsrHandlerTask;       // Handle to IsrHandlerTask()
};

// Base class that has a lock, an ISR task, and a initializer task that can block
class CDeviceDriverIsrInit : public CDeviceDriverIsr
{
  public:
    CDeviceDriverIsrInit(void);
    ~CDeviceDriverIsrInit(void);
    bool IsInitialized(void) {return m_Initialized;}; // Checks if the driver is initialized
  protected: 
    bool m_Initialized; // Signals driver initialization
    UINT32 m_hInitializerTask;       // Handle to InitializerTask()
    // Configure the initializer task, must be called only after the derived class is initialized
    bool ConfigureInitializerTask(UINT16 StackSize, UINT32 Priority, const signed char* pTaskName);
    static void InitializerTaskWrapper(UINT32 Param); // C wrapper for IsrHandlerTask()                            
    virtual void InitializerTask(void) = 0;           // Task handling the ISR event
};

// All data queued using CDeviceDriverQueueTask must be in the following format
struct QueueData {
  UINT8* pData;     // Data ptr
  UINT16 DataSize;  // Data size
  bool   Copied;    // If the data is copied during enqueue
  bool   Free;      // If the dequeue handler has to free the memory afterwards
};

// Base class that has a lock, queue, and a dequeue task
class CDeviceDriverQueueTask : public CDeviceDriver
{
  public:
    CDeviceDriverQueueTask(void);
    ~CDeviceDriverQueueTask(void);
    
    // Must be called by derived classes unless overriden 
    bool BaseInitialize(UINT32 QueueSize, UINT16 StackSize, UINT32 Priority, char* pTaskName);
    
    // Methods for queuing data by copy or reference
    bool EnqueueCopy(UINT8* pData, UINT16 DataSize);
    bool Enqueue(UINT8* pData, UINT16 DataSize, bool Free);
  protected: 
    void DequeueTask(void); // Task that waits on the queue
    static void DequeueTaskWrapper(UINT32 Param); // C wrapper for DequeueTask()                            
    virtual void QueueHandler(QueueData* pQueueData) = 0; // Queue handler
    
    UINT32 m_QueueSize;     // Queue size
    UINT32 m_hQueue;        // Queue handle
    UINT32 m_hDequeueTask;  // Queue task handle
};
#endif  // _C_DEVICE_DRIVER_H_

